(in-package #:cl-fast-behavior-trees)


(define-behavior-tree-node (:repeat
                            :inline t)
    ()
  "Constantly reruns its only child."
  (when (child-completed-p 0)
    (reset-children)
    (when rootp
      (return-from-tree)))
  (deactivate)
  (activate-child 0))

(define-behavior-tree-node (:repeat-until-fail
                            :inline t)
    ()
  "Constantly reruns its only child until it fails."
  (when (child-completed-p 0)
    (if (child-succeeded-p 0)
        (progn
          (reset-children)
          (when rootp
            (return-from-tree)))
        (return (complete-node t))))
  (deactivate)
  (activate-child 0))

(define-behavior-tree-node (:invert
                            :inline t)
    ()
  "Succeeds only if its only child fails and vice versa."
  (when (child-completed-p 0)
    (return (complete-node (not (child-succeeded-p 0)))))
  (deactivate)
  (activate-child 0))

(define-behavior-tree-node (:fallback
                            :inline t)
    ((current 0 :type array-index))
  "Fallback is like an OR logical element. AKA selector."
  (when (child-completed-p current)
    (if (child-succeeded-p current)
        (return (complete-node t))
        (when (>= (incf current) children-count)
          (return (complete-node nil)))))
  (deactivate)
  (activate-child current))

(define-behavior-tree-node (:sequence
                            :inline t)
    ((current 0 :type array-index))
  "Sequence node is like an AND logical element."
  (when (child-completed-p current)
    (if (not (child-succeeded-p current))
        (return (complete-node nil))
        (when (>= (incf current) children-count)
          (return (complete-node t)))))
  (deactivate)
  (activate-child current))

(define-behavior-tree-node (:random
                            :inline t)
    ((probability 0.5 :type single-float))
  "Random node succeeds with given probability (from 0 to 1)."
  (complete-node (< (random 1.0) probability)))

(define-behavior-tree-node (:always-true
                            :inline t)
    ()
  "Unconditionally succeeds."
  (complete-node t))

(define-behavior-tree-node (:always-false
                            :inline t)
    ()
  "Unconditionally fails."
  (complete-node nil))
