(in-package #:cl-fast-behavior-trees)


(ecs:defcomponent behavior-tree-marker
  (type '|| :type symbol))

(defgeneric make-behavior-tree (tree entity &rest initargs))
(defgeneric %delete-behavior-tree (tree entity))

(defgeneric %print-tree (tree entity))
(defgeneric %dump-tree (tree entity))

(defun delete-behavior-tree (entity)
  (%delete-behavior-tree (behavior-tree-marker-type entity) entity))

(defstruct behavior-tree-node-instance
  (type nil :type (or null behavior-tree-node))
  (name "" :type string-designator)
  (options nil :type list)
  (depth 0 :type array-index)
  (children nil :type list)
  parent)

(defun parse-spec (spec &optional
                          (parent nil)
                          (type-counters (make-hash-table :test #'eq))
                          (depth 0))
  (let+ (((type-and-options &rest children) spec)
         (type (if (typep type-and-options 'symbol)
                   type-and-options (first type-and-options)))
         ((&values type* type-exists) (gethash type *node-types*))
         ((&values) (unless type-exists
                      (error "Unknown behavior tree node type: ~a" type)))
         (options (if (typep type-and-options 'symbol)
                      nil (rest type-and-options)))
         (name (or (getf options :name)
                   (symbolicate
                    type
                    (write-to-string
                     (incf (the fixnum (gethash type type-counters 0))))))))
    (let ((node (make-behavior-tree-node-instance :type type*
                                                  :name name
                                                  :options options
                                                  :depth depth
                                                  :parent parent)))
      (setf (behavior-tree-node-instance-children node)
            (mapcar
             (lambda (child-spec)
               (parse-spec child-spec node type-counters (1+ depth)))
             children))
      node)))

(defun flatten-tree (root)
  (let ((children (behavior-tree-node-instance-children root)))
    (append
     (list root)
     (mapcan #'flatten-tree children))))

(declaim (ftype (function (symbol list &key (:debug t)) list)
                %define-behavior-tree))
(defun %define-behavior-tree (tree-name spec &key (debug nil))
  (let+ ((root (parse-spec spec))
         (nodes (flatten-tree root))
         (tree-size (length nodes))
         (node-types (mapcar #'behavior-tree-node-instance-type nodes))
         (entity (intern "ENTITY" *package*))
         (node-dump-format
          "\"~a\"[shape=~:[rect~;oval~],label=\"~a\",style=\"~:[~;filled~]\"];")
         (debug-completion-format
          "~a: ~a node ~a for entity ~a ~:[failed~;succeeded~]")
         (debug-fn (if (eq debug t) (lambda (s) (format t "~a~%" s)) debug))
         ((&flet node-name (node)
            (string-upcase (behavior-tree-node-instance-name node))))
         ((&flet collect-option (accessor)
            (reduce #'union node-types :key accessor :initial-value nil)))
         ((&flet reset-node-exprs (node)
            (let ((type (behavior-tree-node-instance-type node))
                  (node-name (node-name node)))
              (append
               (list
                (symbolicate tree-name :- node-name :-completed) 0
                (symbolicate tree-name :- node-name :-succeeded) 0)
               (mapcan
                (lambda (name default)
                  (list
                   (symbolicate tree-name :- node-name :- name)
                   (if-let (option
                            (getf (behavior-tree-node-instance-options node)
                                  (make-keyword name)))
                     option
                     default)))
                (behavior-tree-node-slot-names type)
                (behavior-tree-node-slot-defaults type))))))
         ((&labels reset-children-exprs (node)
            (loop :for child :in (behavior-tree-node-instance-children node)
                  :nconc (append (reset-node-exprs child)
                                 (reset-children-exprs child)))))
         ((&with-gensyms init-tree run-current-node component-storages)))
    (unless (setp nodes :test #'equal :key #'behavior-tree-node-instance-name)
      (error "Node names are not unique"))
    `(progn
       (ecs:defcomponent ,tree-name
           (current-node 0 :type (integer -1 ,(1- tree-size)))
         ,@(mapcan
            (lambda (n node node-type)
              (declare (type array-index n)
                       (type behavior-tree-node-instance node)
                       (type behavior-tree-node node-type))
              (nconc
               (mapcar
                (lambda (slot-name slot-default slot-type slot-docstring)
                  `(,(symbolicate (node-name node) :- slot-name)
                    ,slot-default
                    :type ,slot-type
                    :documentation ,slot-docstring))
                (behavior-tree-node-slot-names node-type)
                (behavior-tree-node-slot-defaults node-type)
                (behavior-tree-node-slot-types node-type)
                (behavior-tree-node-slot-docstrings node-type))
               (list
                `(,(symbolicate (node-name node) :-active)
                  ,(if (zerop n) 1 0)
                  :type bit)
                `(,(symbolicate (node-name node) :-completed) 0 :type bit)
                `(,(symbolicate (node-name node) :-succeeded) 0 :type bit))))
            (iota (length nodes)) nodes node-types))
       (pushnew ,(make-keyword tree-name) ecs:*skip-printing-components*
                :test #'eq)

       (defun ,init-tree (,entity &rest initargs)
         (apply #',(symbolicate :make- tree-name)
                ,entity
                ,@(loop :for node :in nodes
                        :for node-name := (node-name node)
                        :for options :=
                           (remove-from-plist
                            (behavior-tree-node-instance-options node)
                            :name)
                        :nconc (loop :for (key value) :on options :by #'cddr
                                     :nconc (list
                                             (format-symbol :keyword "~a-~a"
                                                            node-name key)
                                             value)))
                initargs))

       (defun ,(symbolicate :make- tree-name :-behavior-tree) (,entity
                                                               &rest initargs)
         (when (has-behavior-tree-marker-p ,entity)
           (error "The entity ~a already has ~a behavior tree assigned"
                  ,entity (behavior-tree-marker-type ,entity)))
         (make-behavior-tree-marker ,entity :type ',tree-name)
         (apply #',init-tree ,entity initargs))

       (defmethod make-behavior-tree ((tree (eql ',tree-name)) entity
                                      &rest args)
         (apply #',(symbolicate :make- tree-name :-behavior-tree)
                entity args))

       (defun ,(symbolicate :delete- tree-name :-behavior-tree) (entity)
         (,(symbolicate :delete- tree-name) entity)
         (delete-behavior-tree-marker entity))

       (defmethod %delete-behavior-tree ((tree (eql ',tree-name)) entity)
         (,(symbolicate :delete- tree-name :-behavior-tree) entity))

       (defmethod %print-tree ((tree (eql ',tree-name)) entity)
         (with-output-to-string (stream)
           ,@(mapcar
              (lambda (n node next-node node-type)
                (let ((node-name (node-name node)))
                  `(format
                    stream
                    ,(concatenate
                      'string
                      (make-string
                       (* (behavior-tree-node-instance-depth node) 4)
                       :initial-element #\Space)
                      (cond
                        ((zerop n)
                         "─")
                        ((eq (behavior-tree-node-instance-parent node)
                             (and
                              next-node
                              (behavior-tree-node-instance-parent next-node)))
                         "├")
                        (t
                         "└"))
                      "── ~a [~a] ~:[~;*~]~%")
                    ',node-name ',(behavior-tree-node-name node-type)
                    (plusp
                     (,(symbolicate tree-name :- node-name :-active) entity)))))
              (iota (length nodes)) nodes (append (rest nodes) '(nil)) node-types)))

       (defmethod %dump-tree ((tree (eql ',tree-name)) entity)
         (with-output-to-string (stream)
           (format
            stream
            "digraph behavior_tree_~a { node [fontname=\"SourceSans3\"]; "
            entity)
           ,@(mapcar
              (lambda (node node-type)
                (let ((node-name (node-name node))
                      (type (behavior-tree-node-name node-type)))
                  `(progn
                     (format
                      stream ,node-dump-format ,node-name
                      ,(find type
                             '(:repeat :repeat-until-fail :invert :fallback
                               :sequence :parallel)
                             :test #'eq)
                      ,(if-let (named
                                (getf (behavior-tree-node-instance-options node)
                                      :name))
                         (format nil "~:@(~a~)~%~a" named type)
                         node-name)
                      (plusp
                       (,(symbolicate tree-name :- node-name :-active) entity)))
                     ,(when-let (parent
                                 (behavior-tree-node-instance-parent node))
                        `(format stream "\"~a\" -> \"~a\"; "
                                 ,(node-name parent) ,node-name)))))
              nodes node-types)
           (format stream "}")))

       (ecs:defsystem ,(symbolicate :lazy-init- tree-name)
         (:components-ro (fbt:behavior-tree-marker)
          :components-no (,tree-name)
          :when (eq ,(intern "BEHAVIOR-TREE-MARKER-TYPE" *package*)
                    ',tree-name))
         (funcall #',init-tree ,entity))

       (ecs:defsystem ,tree-name
         (:components-ro (,@(collect-option
                             #'behavior-tree-node-options-components-ro))
          :components-rw (,tree-name
                          ,@(collect-option
                             #'behavior-tree-node-options-components-rw))
          :components-no (,@(collect-option
                             #'behavior-tree-node-options-components-no))
          :arguments (,@(collect-option #'behavior-tree-node-options-arguments))
          :initially (progn ,@(mapcar #'behavior-tree-node-options-initially
                                      node-types))
          :finally (progn ,@(mapcar #'behavior-tree-node-options-finally
                                    node-types))
          :when (and ,@(mapcar #'behavior-tree-node-options-when
                               node-types))
          :enable (and ,@(mapcar #'behavior-tree-node-options-enable
                                 node-types))
          :after (,@(collect-option #'behavior-tree-node-options-after))
          :before (,@(collect-option #'behavior-tree-node-options-before)))
         (let ((,component-storages (ecs::storage-component-storages
                                     (the ecs::storage ecs:*storage*))))
           (tagbody
              ,run-current-node
              (case ,(symbolicate tree-name :-current-node)
                ,@(mapcar
                   (lambda (n node type)
                     (let* ((node-name (node-name node))
                            (parent
                              (behavior-tree-node-instance-parent node))
                            (children
                              (behavior-tree-node-instance-children node))
                            (children-count (length children)))
                       `(,n
                         (let+ (((&helper child-completed-p (i)
                                   (case i
                                     ,@(mapcar
                                        (lambda (n child)
                                          (let ((name (node-name child)))
                                            `(,n
                                              (plusp
                                               ,(symbolicate
                                                 tree-name :- name
                                                 :-completed)))))
                                        (iota children-count) children))))
                                ((&helper child-succeeded-p (i)
                                   (case i
                                     ,@(mapcar
                                        (lambda (n child)
                                          (let ((name (node-name child)))
                                            `(,n
                                              (plusp
                                               ,(symbolicate
                                                 tree-name :-
                                                 name :-succeeded)))))
                                        (iota children-count) children))))
                                ((&helper complete (success)
                                   ,@(when debug
                                       `((funcall
                                          ,debug-fn
                                          (format
                                           nil
                                           ,debug-completion-format
                                           ',tree-name
                                           ',(behavior-tree-node-name type)
                                           ',node-name ,entity success))))
                                   (setf
                                    ,(symbolicate tree-name :-current-node)
                                    ,(or (position parent nodes :test #'eq)
                                         -1)
                                    ,(symbolicate
                                      tree-name :- node-name :-active)
                                    0
                                    ,(symbolicate
                                      tree-name :- node-name :-completed)
                                    1
                                    ,(symbolicate
                                      tree-name :- node-name :-succeeded)
                                    (if success 1 0)
                                    ,@(unless (zerop n)
                                        `(,(symbolicate
                                            tree-name :-
                                            (string-upcase
                                             (behavior-tree-node-instance-name
                                              parent))
                                            :-active)
                                          1)))))
                                ((&helper deactivate ()
                                   (setf
                                    ,(symbolicate
                                      tree-name :- node-name :-active)
                                    0)))
                                ((&helper activate-child (i)
                                   (case i
                                     ,@(mapcar
                                        (lambda (n child)
                                          (let ((name (node-name child)))
                                            `(,n
                                              (setf
                                               ,(symbolicate
                                                 tree-name :- name :-active)
                                               1
                                               ,(symbolicate tree-name
                                                             :-current-node)
                                               ,(position child
                                                          nodes
                                                          :test #'eq)))))
                                        (iota children-count) children))))
                                ((&helper reset-children ()
                                   (setf ,@(reset-children-exprs node))))
                                ((&helper reset-tree ()
                                   (setf
                                    ,@(reset-node-exprs root)
                                    ,@(reset-children-exprs root)
                                    ,(symbolicate tree-name :-current-node) 0)))
                                ((&helper return-from-tree ()
                                   (return-from ecs:current-entity)))
                                ((&helper delete-tree ()
                                   (,(symbolicate :delete- tree-name)
                                    ,entity))))
                           ,@(when debug
                               `((funcall
                                  ,debug-fn
                                  (format
                                   nil
                                   "~a: running ~a node ~a for entity ~a"
                                   ',tree-name
                                   ',(behavior-tree-node-name type)
                                   ',node-name ,entity))))
                           (,(behavior-tree-node-function type)
                            ,component-storages
                            ,entity
                            ,(zerop n)
                            ,children-count
                            #'child-completed-p
                            #'child-succeeded-p
                            #'complete
                            #'deactivate
                            #'activate-child
                            #'reset-children
                            #'reset-tree
                            #'return-from-tree
                            #'delete-tree
                            ,@(mapcar
                               (lambda (slot-name)
                                 (let ((accessor-name
                                         (symbolicate
                                          tree-name :- node-name
                                          :- slot-name)))
                                   `(lambda (&optional
                                        (value nil value-supplied-p))
                                      (if value-supplied-p
                                          (setf ,accessor-name value)
                                          ,accessor-name))))
                               (behavior-tree-node-slot-names type))
                            ,@(mapcan
                               (lambda (arg)
                                 (list (make-keyword (first arg))
                                       (first arg)))
                               (behavior-tree-node-options-arguments type)))
                           ,@(when (behavior-tree-node-inline type)
                               `((go ,run-current-node)))))))
                   (iota (length nodes)) nodes node-types))))))))

(defmacro define-behavior-tree (name spec &key debug)
  (%define-behavior-tree name spec :debug debug))
